const app = new Vue({
    el: '#app',
    data() {
        return {
            servers: [],
            templates: [],
            showPortModalFlag: false,
            showTemplateModal: false,
            showBulkIpModal: false,
            showDetailsModal: false,
            selectedServer: null,
            newPort: '',
            newTemplate: {
                port: '',
                ip: '',
                username: '',
                password: '',
                q_port: '',
                name: '',
                description: ''
            },
            bulkIpChange: {
                oldIp: '',
                newIp: '',
                affected: []
            },
            loading: {
                port: false,
                template: false,
                bulkIp: false,
                delete: false,
                edit: false,
                apiSettings: false
            },
            error: null,
            iframeUrl: '',
            serverPage: 1,
            templatePage: 1,
            perPage: 10,
            totalServers: 0,
            totalTemplates: 0,
            alert: {
                show: false,
                message: '',
                type: 'success', // success, error, warning
                timeout: null
            },
            search: {
                server: '',
                template: ''
            },
            showEditModal: false,
            showDeleteModal: false,
            editServer: {
                id: null,
                port: '',
                ip: '',
                q_username: '',
                q_password: '',
                q_port: '',
                q_id: '',
                updatePortOnTs3: false
            },
            deleteServerId: null,
            showApiSettingsModal: false,
            apiSettings: {
                admin_api_key: '',
                api_key: '',
                ts_default_server_name: ''
            },
            stats: {
                total_users: 0,
                total_servers: 0,
                servers_by_ip: []
            }
        }
    },
    computed: {
        totalServerPages() {
            const filteredCount = this.filteredServers.length;
            return Math.ceil(filteredCount / this.perPage);
        },
        totalTemplatePages() {
            return Math.ceil(this.totalTemplates / this.perPage);
        },
        paginatedServers() {
            const filtered = this.filteredServers;
            const start = (this.serverPage - 1) * this.perPage;
            const end = start + this.perPage;
            return filtered.slice(start, end);
        },
        paginatedTemplates() {
            return this.filteredTemplates;
        },
        filteredServers() {
            if (!this.search.server) {
                return this.servers;
            }
            
            const searchLower = this.search.server.toLowerCase();
            return this.servers.filter(server => {
                return server.port.toString().includes(searchLower) ||
                       server.q_ip.toLowerCase().includes(searchLower) ||
                       server.q_username.toLowerCase().includes(searchLower);
            });
        },
        filteredTemplates() {
            if (!this.search.template) return this.templates;
            
            const searchLower = this.search.template.toLowerCase();
            return this.templates.filter(template => {
                return template.name.toLowerCase().includes(searchLower) ||
                       template.description.toLowerCase().includes(searchLower);
            });
        }
    },
    watch: {
        'search.server'() {
            // Arama yapıldığında ilk sayfaya dön
            this.serverPage = 1;
        }
    },
    mounted() {
        if (vueData && vueData.servers) {
            this.servers = vueData.servers;
            this.totalServers = vueData.servers.length;
        }
        this.refreshServerList();
        this.getTemplates();
    },
    methods: {
        // Alert gösterme metodu
        showAlert(message, type = 'success') {
            if (this.alert.timeout) {
                clearTimeout(this.alert.timeout);
            }
            
            this.alert.message = message;
            this.alert.type = type;
            this.alert.show = true;
            
            this.alert.timeout = setTimeout(() => {
                this.alert.show = false;
            }, 3000);
        },

        // Toplu IP değiştirme
        async changeBulkIp() {
            try {
                this.loading.bulkIp = true;
                const response = await axios.post('ajax/change_bulk_ip.php', this.bulkIpChange);
                
                if (response.data.status) {
                    this.showAlert(response.data.message, 'success');
                    await this.refreshServerList();
                    setTimeout(() => {
                        this.showBulkIpModal = false;
                        this.bulkIpChange = {
                            oldIp: '',
                            newIp: '',
                            affected: []
                        };
                    }, 1000);
                } else {
                    throw new Error(response.data.message);
                }
            } catch (error) {
                this.showAlert(error.message || 'IP değiştirme işlemi başarısız', 'error');
            } finally {
                this.loading.bulkIp = false;
            }
        },

        // Template silme
        async deleteTemplate(templateId) {
            if (!confirm('Bu template\'i silmek istediğinizden emin misiniz?')) {
                return;
            }

            try {
                this.loading.delete = true;
                const response = await axios.post('ajax/delete_template.php', {
                    template_id: templateId
                });
                
                if (response.data.status) {
                    this.showAlert(response.data.message, 'success');
                    await this.getTemplates();
                } else {
                    throw new Error(response.data.message);
                }
            } catch (error) {
                this.showAlert(error.message || 'Template silinirken hata oluştu', 'error');
            } finally {
                this.loading.delete = false;
            }
        },

        // Template oluşturma
        async createTemplate() {
            try {
                this.loading.template = true;
                const response = await axios.post('ajax/create_template.php', {
                    name: this.newTemplate.name,
                    ip: this.newTemplate.ip,
                    port: this.newTemplate.port,
                    q_username: this.newTemplate.username,
                    q_password: this.newTemplate.password,
                    q_port: this.newTemplate.q_port
                });
                
                if (response.data.status) {
                    this.showAlert(response.data.message, 'success');
                    await this.getTemplates();
                    setTimeout(() => {
                        this.showTemplateModal = false;
                        this.newTemplate = {
                            port: '',
                            ip: '',
                            username: '',
                            password: '',
                            q_port: '',
                            name: '',
                            description: ''
                        };
                    }, 1000);
                } else {
                    throw new Error(response.data.message);
                }
            } catch (error) {
                this.showAlert(error.message || 'Template oluşturulurken hata oluştu', 'error');
            } finally {
                this.loading.template = false;
            }
        },

        // Template listesini al
        async getTemplates() {
            try {
                const response = await axios.get('ajax/get_templates.php', {
                    params: {
                        page: this.templatePage,
                        per_page: this.perPage
                    }
                });
                if (response.data.status) {
                    this.templates = response.data.templates;
                    this.totalTemplates = response.data.total;
                }
            } catch (error) {
                console.error('Template listesi alınamadı:', error);
            }
        },

        // Sunucu detaylarını göster ve iframe URL'ini al
        async showServerDetails(server) {
            try {
                const response = await axios.post('ajax/get_iframe_url.php', {
                    server_id: server.id
                });
                
                if (response.data.status) {
                    this.iframeUrl = response.data.url;
                    this.selectedServer = server;
                    this.showDetailsModal = true;
                } else {
                    throw new Error(response.data.message);
                }
            } catch (error) {
                this.showAlert(error.message || 'Sunucu detayları alınamadı', 'error');
            }
        },

        // Sunucu listesini yenile
        async refreshServerList() {
            try {
                const response = await axios.get('ajax/get_servers.php');
                if (response.data.status) {
                    this.servers = response.data.servers;
                    this.totalServers = response.data.servers.length;
                }
            } catch (error) {
                console.error('Sunucu listesi alınamadı:', error);
                this.showAlert('Sunucu listesi alınamadı', 'error');
            }
        },

        // Çıkış işlemi
        async logout() {
            try {
                await axios.post('ajax/logout.php');
                window.location.href = 'login.php';
            } catch (error) {
                this.showAlert('Çıkış yapılırken hata oluştu', 'error');
            }
        },

        // Düzenleme modalını aç
        openEditModal(server) {
            console.log('Server data:', server);
            this.editServer = {
                id: server.id,
                port: server.port,
                ip: server.q_ip,
                q_username: server.q_username,
                q_password: server.q_password,
                q_port: server.q_port,
                q_id: server.q_id,
                updatePortOnTs3: false
            };
            this.showEditModal = true;
        },

        // Sunucu düzenleme
        async updateServer() {
            try {
                this.loading.edit = true;
                const response = await axios.post('ajax/update_server.php', this.editServer);
                
                if (response.data.status) {
                    this.showAlert(response.data.message, 'success');
                    await this.refreshServerList();
                    this.showEditModal = false;
                } else {
                    throw new Error(response.data.message);
                }
            } catch (error) {
                this.showAlert(error.message || 'Sunucu güncellenirken hata oluştu', 'error');
            } finally {
                this.loading.edit = false;
            }
        },

        // Silme modalını aç
        openDeleteModal(serverId) {
            this.deleteServerId = serverId;
            this.showDeleteModal = true;
        },

        // Sunucu silme
        async deleteServer() {
            try {
                this.loading.delete = true;
                const response = await axios.post('ajax/delete_server.php', {
                    server_id: this.deleteServerId
                });
                
                if (response.data.status) {
                    this.showAlert(response.data.message, 'success');
                    await this.refreshServerList();
                    this.showDeleteModal = false;
                } else {
                    throw new Error(response.data.message);
                }
            } catch (error) {
                this.showAlert(error.message || 'Sunucu silinirken hata oluştu', 'error');
            } finally {
                this.loading.delete = false;
            }
        },

        // API ayarları modalını aç
        async openApiSettings() {
            try {
                const response = await axios.get('ajax/get_api_settings.php');
                if (response.data.status) {
                    this.apiSettings = response.data.settings;
                    this.showApiSettingsModal = true;
                }
            } catch (error) {
                this.showAlert('API ayarları alınamadı', 'error');
            }
        },

        // API ayarlarını güncelle
        async updateApiSettings() {
            try {
                this.loading.apiSettings = true;
                const response = await axios.post('ajax/update_api_settings.php', this.apiSettings);
                
                if (response.data.status) {
                    this.showAlert(response.data.message, 'success');
                    this.showApiSettingsModal = false;
                } else {
                    throw new Error(response.data.message);
                }
            } catch (error) {
                this.showAlert(error.message || 'API ayarları güncellenirken hata oluştu', 'error');
            } finally {
                this.loading.apiSettings = false;
            }
        },

        // İstatistik sayfasına yönlendir
        goToStats() {
            window.location.href = 'stats.php';
        }
    }
}); 